// offscreen.js
// オフスクリーンで音声を再生するためのスクリプト

// --- State Management ---
let lofiState = {
  isPlaying: false,
  fileName: null,
  volume: 50,
  trackIndex: 0,
};

let ambientState = {
  volumes: {} // { 'fileName.mp3': 50, ... }
};

let lofiAudio = null;
let ambientAudios = new Map();


// --- Message Listener ---
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  const handled = new Set([
    'playLofi', 'pauseLofi', 'stopLofi', 'setLofiVolume',
    'playAmbient', 'setAmbientVolume', 'stopAmbient', 'stopAllAmbient',
    'playNotificationSound', 'playTestSound', 'playCustomSound'
  ]);

  // 未対応のメッセージは無視して他のリスナーに委ねる
  if (!message || !handled.has(message.type)) {
    return false; // 同期レスポンスも返さず、非ハンドリングを示す
  }

  (async () => {
    try {
      let newState;
      switch (message.type) {
        // Lofi
        case 'playLofi':
          newState = await playLofi(message.fileName, message.volume, message.trackIndex);
          sendResponse({ success: true, newState });
          break;
        case 'pauseLofi':
          newState = pauseLofi();
          sendResponse({ success: true, newState });
          break;
        case 'stopLofi':
          newState = stopLofi();
          sendResponse({ success: true, newState });
          break;
        case 'setLofiVolume':
          newState = setLofiVolume(message.volume);
          sendResponse({ success: true, newState });
          break;

        // Ambient
        case 'playAmbient':
          newState = await playAmbient(message.fileName, message.volume);
          sendResponse({ success: true, newState });
          break;
        case 'setAmbientVolume':
          newState = setAmbientVolume(message.fileName, message.volume);
          sendResponse({ success: true, newState });
          break;
        case 'stopAmbient':
          newState = stopAmbient(message.fileName);
          sendResponse({ success: true, newState });
          break;
        case 'stopAllAmbient':
          newState = stopAllAmbient();
          sendResponse({ success: true, newState });
          break;

        // Notifications
        case 'playNotificationSound':
          await playNotificationSound();
          sendResponse({ success: true });
          break;
        case 'playTestSound':
          await playTestSound(message.soundFile, message.volume);
          sendResponse({ success: true });
          break;
        case 'playCustomSound':
          await playCustomSound(message.sound, message.volume, message.loop);
          sendResponse({ success: true });
          break;
      }
    } catch (error) {
      console.error(`Error processing message ${message.type}:`, error);
      sendResponse({ success: false, error: error.message });
    }
  })();

  return true; // 非同期レスポンスを返すことを宣言
});


// --- Lo-Fi Music Functions ---
async function playLofi(fileName, volume, trackIndex) {
  if (lofiAudio) {
    lofiAudio.pause();
    lofiAudio.currentTime = 0;
    lofiAudio = null;
  }

  lofiAudio = new Audio(`audio/LOFI/${fileName}`);
  lofiAudio.volume = Math.max(0, Math.min(1.0, volume / 100));
  lofiAudio.loop = true;
  await lofiAudio.play();

  lofiState = { isPlaying: true, fileName, volume, trackIndex };
  return lofiState;
}

function pauseLofi() {
  if (lofiAudio) {
    lofiAudio.pause();
  }
  lofiState.isPlaying = false;
  return lofiState;
}

function stopLofi() {
  if (lofiAudio) {
    lofiAudio.pause();
    lofiAudio.currentTime = 0;
    lofiAudio = null;
  }
  lofiState = { ...lofiState, isPlaying: false, fileName: null };
  return lofiState;
}

function setLofiVolume(volume) {
  lofiState.volume = volume;
  if (lofiAudio) {
    lofiAudio.volume = Math.max(0, Math.min(1.0, volume / 100));
  }
  return lofiState;
}


// --- Ambient Sound Functions ---
async function playAmbient(fileName, volume) {
  if (volume === 0) {
    return stopAmbient(fileName);
  }

  if (ambientAudios.has(fileName)) {
    // 既に存在する場合は音量のみ更新
    const audio = ambientAudios.get(fileName);
    audio.volume = Math.max(0, Math.min(1.0, volume / 100));
  } else {
        // 新しい音声を作成して再生
        try {
          const audio = new Audio(`audio/ES/${fileName}`);
          audio.loop = true;
          audio.volume = Math.max(0, Math.min(1.0, volume / 100));

          // まずマップに登録しておくことで、再生完了前に stopAmbient が呼ばれても
          // 正しく pause() できるようにする
          ambientAudios.set(fileName, audio);

          // play() は非同期で完了を待つ。失敗した場合はマップをクリーンアップ
          await audio.play().catch(err => {
            ambientAudios.delete(fileName);
            throw err;
          });

        } catch (error) {
          console.error(`Failed to play ambient sound ${fileName}:`, error);
          throw error;
        }
  }
  
  ambientState.volumes[fileName] = volume;
  return ambientState;
}

function setAmbientVolume(fileName, volume) {
  if (volume === 0) {
    return stopAmbient(fileName);
  }
  
  if (ambientAudios.has(fileName)) {
    const audio = ambientAudios.get(fileName);
    audio.volume = Math.max(0, Math.min(1.0, volume / 100));
  }
  ambientState.volumes[fileName] = volume;
  return ambientState;
}

function stopAmbient(fileName) {
  if (ambientAudios.has(fileName)) {
    const audio = ambientAudios.get(fileName);
    try {
      audio.pause();
      audio.currentTime = 0;
    } catch (error) {
      console.error(`Error stopping ambient sound ${fileName}:`, error);
    }
    ambientAudios.delete(fileName);
  }
  delete ambientState.volumes[fileName];
  return ambientState;
}

function stopAllAmbient() {
  ambientAudios.forEach((audio) => {
    audio.pause();
    audio.currentTime = 0;
  });
  ambientAudios.clear();
  ambientState.volumes = {};
  return ambientState;
}


// --- Notification Sound Functions ---
async function playNotificationSound() {
  const audio = new Audio('audio/notify_simple.mp3');
  await audio.play();
}

async function playTestSound(soundFile, volume = 1.0) {
  const audio = new Audio(`audio/${soundFile}`);
  audio.volume = Math.max(0, Math.min(1.0, volume));
  await audio.play();
}

async function playCustomSound(soundFile, volume = 1.0, loop = false) {
    const audio = new Audio(`audio/${soundFile}`);
    audio.volume = Math.max(0, Math.min(1.0, volume / 100));
    audio.loop = loop;
    await audio.play();
}
